$TITLE UNI-CGE.gms A CGE Model to Teach General Equilbrium Theory and Policy Analysis

$OffSymXRef
$offSymList

*INTRODUCTION

$ontext
UNI-CGE.gms, A CGE Model to Teach General Equilbrium Theory and Policy Analysis

Programmed by:
 Karen Thierfelder (Professor, US Naval Academy)
 email: Karen.Thierfelder@gmail.com
 
 Mary Burfisher (Senior Education Advisor, the Global Trade Analysis Project, GTAP)
 email: Burfisher@gmail.com

This program starts from the teaching version of the IFPRI standard model, adapting it to
teach general equilibrium theory and policy analysis at an intuitive leve.

The model code and simulation files complement teaching material available at Open University OpenLearn,
"Fundamentals of Computable General Equilibrium Modeling" developed by
Mary Burfisher and Karen Thierfelder

References:

Lofgren, Hans, Rebecca Harris and Sherman Robinson 2002.
"A standard computable general equilibrium (CGE) model in GAMS," International
Food Policy Research Institute (IFPRI).
 
Robinson, Sherman and Karen Thierfelder. 2023. "Teaching Model version of
the IFPRI Standard Model," International Food Policy Research Institute (IFPRI).

Robinson, Sherman, Maureen Kilkenny, and Kenneth Hanson. 1990. "The USDA/ERS Computable
General Equilibrium (CGE) Model of the United States," U.S. Department of Agriculture, Economic
Research Service, Agriculture and Rural Economy Division.
 
$OffText

$onText
TABLE OF CONTENTS

1. SETS AND INPUT SAM DATA

2. MODEL PARAMETERS
   In this section of the code, we identify parameters used in the model, by names and definitiona.
   Parameters are listed alphabetically.
   
3. MODEL VARIABLES
   In this section of the code, we identify variables by names and definitions. Variables are listed
   alphabetically.
   
4. MODEL EQUATIONS
   In this section of the code, we define the equations, assign functions, and list the equations
   in the model.
   
   Equations are organized in the following sections and are alphabetical within each section.
   The model, called UNI, is asystem of equations. The list of equations in the model and
   the associated variable for that equation, follows the equation assignment. 
   
     DEMAND
     Consumption choices by  households, government and investors
     
     PRODUCTION
     Output, Intermediate Input Demand and Factor Demand
     
     TRADE
     Export Supply and Import Demand
     
     EQUILIBRIUM CONSTRAINTS
     Market Clearing Conditions for Commodities, Factors, Interactions with the Rest of the World
     
     PRICE INDICES
     Model Numeraire Options
  
5. PARAMETER ESTIMATION AND CALIBRATION
    Given the transactions data in the Social Accounting Matrix (SAM), the selected functions used
    to represent behavior, and the elasticities of substitution or transformation, we calculate the
    the parameters in the functions. The code is in an include file, 1Calibaration-UNI.inc

6. VARIABLE INITIALIZATION
    To make it easier for the model to solve, we initialize all variables at the base level.

7. MODEL CLOSURES
    The model closure defines how the economy adjusts to a policy shock. In all cases, one variable
    is flexible, another variable is held constant (fixed). The closures are described below. In
    the code, the user assigns a number associated with the desired specification.

   SAVINGS-INVESTMENT CLOSURE
      Savings-driven investment: flexible investment spending, fixed household savings rate
      Investment-driven savings: fixed investment spending, flexible household savings rate
      
   GOVERNNMENT CLOSURE
      fixed government expenditure, fixed tax rates, flexible government savings
      flexible government expenditure, fixed tax rates, fixed government savings
      fixed government expenditure, flexible tax rate for income taxes, fixed government savings
      
   FOREIGN EXCHANGE MARKET CLOSURE
      fixed foreign savings, flexible exchange rate
      flexible foreign savings, fixed exchange rate
      
   FACTOR MARKETS
      full employment: fixed factor supply, flexible payment to factor (wage)
      unemployment: flexible factor supply, fixed payment to factor (wage)
      activity-specific factors: fixed factor use by activity,
      flexible activity-specific factor payment
      
8. MODEL SOLUTION
      Run the model using the PATH solver in GAMS.
      
9. REPORT BASE MODEL
     This is an include file to UNI.GMS. It produces the following reports
     on the economy based on data for the base solution (with the name of the
     relevant parameter in parentheses):
     1. Economic structure table by activity   (STRUCBASE1)
     2. Economic structure table by commoidity (STRUCBASE2)
     3. GDP table (GDPBASE)
     4. Macro SAM (MACROSAM)
     5. Input shares, factor wages, and elasticities (FACTAB1 - 4)
     6. Household incomes (HHDTAB1 - 2)
     7. Regional export shares (REGETAB)
     8. Regional import shares (REGMTAB)

     For STRUCBASE, note that both activity and commodity data is presented
     under commodity headings.
     
$offText

*SOFTWARE LICENSE
$OnText

This programme is licenced under the terms of the Creative Commons

Attribution-ShareAlike 4.0 International (CC BY-SA 4.0) license.

(see https://creativecommons.org/licenses/by-sa/4.0/
 and https://creativecommons.org/licenses/by-sa/4.0/legalcode)

You are free to:

Share � copy and redistribute the material in any medium or format
Adapt � remix, transform, and build upon the material for any purpose,
        even commercially.

Under the following terms:

Attribution � You must give appropriate credit, provide a link to the
         license, and indicate if changes were made. You may do so
         in any reasonable manner, but not in any way that suggests
         the licensor endorses you or your use.

ShareAlike � If you remix, transform, or build upon the material, you
         must distribute your contributions under the same license
         as the original.

No additional restrictions � You may not apply legal terms or technological
         measures that legally restrict others from doing anything the
         license permits.

The licensor cannot revoke these freedoms as long as you follow the
         license terms.
$OffText


*1. SETS AND INPUT SAM DATA
$OnFold

SETS

 AC       global set 
 ACNT(AC) all elements in AC except total
 A(AC)    activities
 ACES(A)  CES production function
 ACD(A)   Cobb Douglas production function
 C(AC)    commodities
 CD(AC)   commodities with domestic sales of output
 CDN(C)   commodities without domestic sales of output Not allowed in this version
 CED(C)   commodities with export demand equations
 CEDN(C)  commodities without export demand equations
 CE(C)    exported commodities
 CEN(C)   non-exported commodities
 CM(C)    imported commodities
 CMN(C)   non-imported commodities
 F(AC)    factors
 FLAB(F)  labor factors
 FSF(F)   activity specific factors
 FUE(F)   unemployed factors
 AFA(F,A) factor F is used in production of activity A
 TFF(AC)  factor use taxes
 INS(AC)  institutions
 H(INS)   households
 

* Mapping sets
 MapF2TFF(F,TFF)  map factors to factor use taxes
   ;

 ALIAS (AC,ACP), (A,AP), (C,CP), (F,FP), (INS,INSP), (TFF,TFFP), (H,HP), (ACNT,ACNTP);
 
Parameter
* Parameters for data entry

 ESUBQ(C)      elasticity of substitution bt. dom goods and imports for C
 ESUBVA(A)     elasticity of substitution bt. factors for CES production
 EDE(c)        export demand elasticity
 ETRAX(C)      elasticity of transformation bt. dom sales and exports for C
 FRISCH0(H)    Frisch parameter for LES system
 LESELAS0(C,H) income demand elasticities for LES system
 SAM(AC,AC)    Social Accounting Matrix SAM
 samfac        Factor by which to scale the SAM
 testQD(C)     test for zero domestic sales of domestic output
;

*Read SAM data, all Sets and all elasticities from an Excel file
$ontext

 Define the excel country data file using the $setglobal command, where "CNTY" is the
 country code assigned to your SAM. 
 
 The default is to load the USA333 SAM, used for demonstration in the course.
 For example, to change the country data file to Turkey (TUR), SAM-TUR.xlsx, replace
 USA333 with TUR.
 
 Country SAMS available:
 
 USA333  SAM used for demonstration in the course
 
             Cameroon
             Ethiopia
             Malawi
             Zimbabwe
             Lebanon
             Bangladesh
             Benin
             Ghana
             Algeria
             Egypt
             Rest East Africa (Somalia)
 
$offText

$onEmbeddedCode Connect:
- ExcelReader:
    file: SAM-usa333.xlsx
    index: index!A4
    
- GDXWriter:  # write to GDX
    file: model-data.gdx
- GAMSWriter:  # write to GAMS
    symbols: all
    
$offEmbeddedCode

*Define ACNT to be AC without entry "Total"
 ACNT(AC) = YES; ACNT('TOTAL') = NO; ALIAS(ACNT,ACNTP);

* Assign subsets and dynamic sets based on read in data
*Sets based on input data
 CM(C)$SAM("ROW",C) = YES ;
 CE(C)$SAM(C,"ROW") = YES ;
*CD is the set for commodities with domestic sales of domestic output
*i.e., for which (value of sales at producer prices)
*              > (value of exports at producer prices)
 testQD(C) = (SUM(A, SAM(A,C))) - (SAM(C,"ROW") - SAM('EXPTAX',C));
 CD(C)     = YES$[testQD(C) GT 1e-6 ] ;
                                    
 CDN(C)$(NOT CD(C)) = YES ;
 
* Commodities with export demand equations
 CED(C) = YES$(EDE(C) GT 0) ;
 CEDN(C) = NOT CED(C) ; 

*Commodities with no exports or no imports 
 CEN(C)$(NOT CE(C)) = YES ;
 CMN(C)$(NOT CM(C)) = YES ;
 
*Set presence of factor F in production of activity A

 AFA(F,A)$SAM(F,A)       = YES ;
 
*Scaling factor for scaling the SAM. Could be read from the spreadsheet.   
 samfac = 1 ;

 SAM(AC,ACP) = SAM(AC,ACP)/samfac ;
 
*Display elasticities
Display ESUBQ, ESUBVA,EDE,ETRAX,FRISCH0, LESELAS0;

$OffFold

*2. MODEL PARAMETERS
* Names and Definitions
$OnFold

PARAMETERS
 ad(A)       efficiency parameter in the production fn for a 
 alpha(F,A)  share of value-added to factor f in activity a
 alphaa(A)   Constant in CES production function 
 alphaq(C)   Armington function shift parameter for commodity c
 alphax(C)   CET function shift parameter for commodity c
 beta(C,H)   share of household consumption spending on commodity c
 cwts(C)     consumer commodity c weights in the CPI
 dwts(C)     producer domestic sales weights for DPI
 deltaa(F,A) CES production function share parameter
 deltaq(C)   Armington function share parameter for commodity c     
 deltax(C)   CET function share parameter for commodity 
 econst(c)   constant for export demand equations
 gamma(C,H)  subsist cons of marketed com c for hhd h
 ica(C,A)    qnty of c as intermediate input per unit of activity a
 mpsbar(H)   base value of mps for households
 mps01(H)    0 if no mps adjustment for household H 1 if adjustment
 pwm(C)      import price for c (foreign currency)
 pwse(C)     world price of export substitutes
 qgbar(C)    government demand for commodity c in base year
 qinvbar(C)  base-year qnty of investment demand for commodity c
 rhoa(A)     Production function exponent for activity A 
 rhoq(C)     Armington function exponent for commodity c
 rhox(C)     CET function exponent for commodity c
 shry(H,F)   share for household h in the income of factor f
 theta(A,C)  yield of output c per unit of activity a
 wfa(F,A)    wage for factor f in activity a (only for calibration)
 ;
 
$OffFold

*3. MODEL VARIABLES
* Names and definitions - listed alphabetically
$OnFold
VARIABLES
 CPI         consumer price indexGDPTAB  Items in GDP table
 DPI         producer price index
 EG          government expenditures
 EH(H)       aggregate household consumption expenditure
 ETAX        export tax revenue
 EXR         exchange rate (dom. currency per unit of for. currency)
 FSAV        foreign savings (in foreign currency units)
 FTAX        factor use tax revenue
 GADJ        government consumption adjustment factor
 GOVSHR      government consumption as share of total absorption
 GSAV        government savings
 HTAX        household income tax revenue
 IADJ        investment adjustment factor
 INVSHR      investment as share of total absorption
 MPS(H)      marginal (and average) propensity to save for household h
 MPSADJ      household MPS adjustment factor
 MTAX        import tax revenue
 PA(A)       price of activity a
 PD(C)       domestic price of domestic output c 
 PE(C)       export price for c (domestic currency) 
 PM(C)       import price for c (domestic currency)
 PQ(C)       composite commodity price for c
 PTAX        production tax revenue
 PVA(A)      value-added price for activity a
 PWE(C)      export price for c (foreign currency)
 PX(C)       producer price for commodity c
 QA(A)       level of activity a
 QD(C)       quantity sold domestically of domestic output c
 QE(C)       quantity of exports for commodity c
 QF(F,A)     quantity demanded of factor f from activity a
 QFS(F)      supply of factor f
 QG(C)       govt demand for commodities
 QH(C,H)     quantity consumed of commodity c by household h
 QINTCA(C,A) qnty of commodity c as intermediate input to activity a
 QINV(C)     quantity of investment demand for commodity c
 QM(C)       quantity of imports of commodity c
 QQ(C)       quantity of goods supplied domestically (composite supply)
 QX(C)       quantity of domestic output of commodity c
 STAX        commodity sales tax revenue
 TABS        total absorption
 TA(A)       rate of tax on producer gross output value 
 TE(C)       rate of tax on exports
 TFA(f,a)    rate of tax on factor use by activity
 TM(C)       rate of import tariff 
 TQ(C)       rate of sales taxes 
 TY(H)       rate of income tax for household h  
 WALRAS      dummy variable (zero at equilibrium)
 WF(F)       average price of factor f
 WFDIST(F,A) wage distortion factor for factor f in activity a
 YF(F)       factor income
 YG          government revenue
 YH(H)       income of household h

 ;
 
$OffFold

*4. MODEL EQUATIONS
* Define Equations
EQUATIONS
* DEMAND
* Consumption choices by  households, government and investors
$OnFold
* Household Demand 
 EHDEF(H)       aggregate household expenditure on commodities
 HHDEM(C,H)     consumption demand for household h & commodity c
 HHDINC(H)      income of household h
 MPSDEF(H)      adjust the marginal propensity to save - MPS
 YFDEF(F)       factor incomes
 
* Government Demand
 EGDEF          total government expenditures
 GDABEQ         government consumption share in absorption
 YGDEF          total government income
 GOVBAL         government balance
 GOVDEM(C)      government demand for commodities
 
* Government tax revenue
 ETAXEQ         export tax revenue
 FTAXEQ         factor use tax revenue
 HTAXEQ         household income tax revenue
 MTAXEQ         import tax revenue
 PTAXEQ         production tax revenue
 STAXEQ         commodity sales tax revenue
 
* Investment Demand
 INVABEQ        investment share in absorption 
 INVDEM(C)      fixed investment demand
 
* Total
 TABSEQ         total absorption
 
* Commodity price equation 
 PQDEF(C)       absorption for commodity c
$OffFold

* PRODUCTION
* Output, Intermediate Input Demand and Factor Demand
$OnFold
*Quantity Equations
 FACDEM(F,A)    CES demand for factor f from activity a 
 INTDEM(C,A)    intermediate demand for commodity c from activity a
 OUTPUTFN(C)    output of commodity c
 PRODFN(A)      CES production function for activity a
 
* Price Equations
 PADEF(A)       output price for activity a
 PVADEF(A)      value-added price for activity a
 PXDEF(C)       output value for commodity c
$offFold

* TRADE
* Export Supply and Import Demand
$OnFold
* Exports
 CET(C)         output transformation (CET) function
 CET2A(C)       domestic sales and no exports
 CET2B(C)       no domestic sales and exports 
 ESUPPLY(C)     export-domestic supply ratio for commodity c
 EDEMAND(C)     export demand
 PEDEF(C)       domestic export price for 

* Imports
 ARMINGTON(C)   composite commodity aggregation function
 ARMINGTON2A(C) supply for comm with D and no M
 ARMINGTON2B(C) supply for comm with M and no D
 COSTMIN(C)     import-domestic demand ratio for commodity c
 PMDEF(C)       domestic import price 
$OffFold

* EQUILIBRIUM CONSTRAINTS
* Market Clearing Conditions for Commodities, Factors, Interactions with the Rest of the World
* and Savings
$OnFold
 COMEQUIL(C)    composite commodity market equilibrium
 CURACCBAL      current account balance (of RoW) 
 FACEQUIL(F)    factor market equilibrium  
 SAVINVBAL      savings-investment balance
$offFold
 
* PRICE INDICES
* Model Numeraire Options 
$OnFold 
 CPIDEF         definition of consumer price index
 DPIDEF         definition of producer price index
 ;
 
$OffFold 
 
* Assign equations
$OnFold
*PRICE BLOCK
 PMDEF(C)$CM(C).. PM(C) =E= (1 + TM(C))*EXR*pwm(C);

 PEDEF(C)$CE(C).. PE(C) =E= (1 - TE(C))*EXR*PWE(C);
 
 PQDEF(C)$(CD(C) OR CM(C))..
                  PQ(C)*(1 - TQ(c))*QQ(C) =E= PD(C)*QD(C) + PM(C)*QM(C);

 PXDEF(C)..       PX(C)*QX(C) =E= PD(C)*QD(C) + (PE(C)*QE(C))$CE(C);

 PADEF(A)..       PA(A) =E= SUM(C, PX(C)*theta(A,C));

 PVADEF(A)..      PVA(A) =E= PA(A)*(1-TA(A)) - SUM(C, PQ(C)*ica(C,A));

*PRODUCTION AND COMMODITY BLOCK

* CES production functions
* Note - ELASTVA cannot equal one to use CES function
 PRODFN(A)..          QA(A) =E=
                       alphaa(A)*(SUM(F$AFA(F,A), deltaa(F,A)*QF(F,A)**(-rhoa(A)) ))                      
                       **(-1/rhoa(A)) ;
                       
 FACDEM(F,A)$AFA(F,A)..
                       WF(F)*WFDIST(F,A)*(1+TFA(F,A)) =E= 
                       PVA(A)*QA(A)*(SUM(FP$deltaa(FP,A), 
                       deltaa(FP,A)*QF(FP,A)**(-rhoa(A))))**(-1)
                      *deltaa(F,A)*QF(F,A)**(-rhoa(A)-1) ;
                      
 INTDEM(C,A)..         QINTCA(C,A) =E= ica(C,A)*QA(A);

 OUTPUTFN(C)..         QX(C) =E= SUM(A, theta(A,C)*QA(A));

 ARMINGTON(C)$(CM(C) AND CD(C))..  QQ(C) =E= alphaq(C)*(deltaq(C)*QM(C)**(-rhoq(C))
                       + (1-deltaq(C))*QD(C)**(-rhoq(C)))**(-1/rhoq(C));

 COSTMIN(C)$(CM(C) AND CD(C))..    QM(C) =E=
                       QD(C)*( (PD(C)/PM(C))
                       *(deltaq(C)/(1-deltaq(C))) )**(1/(1 + rhoq(C)));
 
 ARMINGTON2A(C)$(CD(C)  AND CMN(C))..  QQ(C) =E= QD(C);
                       
 ARMINGTON2B(C)$(CDN(C) AND CM(C))..   QQ(C) =E= QM(C);
                       
 CET(C)$(CE(C) AND CD(C))..      QX(C) =E= alphax(C)*(deltax(C)*QE(C)**rhox(C)
                                     + (1-deltax(C))*QD(C)**rhox(C) )**(1/rhox(C)); 

 CET2A(C)$(CD(C)  AND CEN(C))..  QX(C) =E= QD(C) ;
                       
 CET2B(C)$(CDN(C) AND CE(C))..   QX(C) =E= QE(C);

 ESUPPLY(C)$(CD(C) AND CE(C))..  QE(C) =E= QD(C)*( PE(C)/PD(C)
                                *(1-deltax(C))/deltax(C) )**(1/(rhox(C)-1) );
                            
 EDEMAND(c)$CED(C)..
                   QE(c) =E= econst(c)*((PWE(c)/pwse(c))**(-ede(c))) ;                                      
                                
*INSTITUTION BLOCK

 YFDEF(F)..      YF(F) =E= SUM(A, WF(F)*WFDIST(F,A)*QF(F,A));
 
 HHDINC(H)..     YH(H) =E= SUM(F, shry(H,F)*YF(F)) ;
                              
 EHDEF(H)..      EH(H) =E= (1 - MPS(H))*(1 - TY(H))*YH(H) ;

 HHDEM(C,H)$(CD(C) OR CM(C)).. PQ(C)*QH(C,H) =E=
                 PQ(C)*gamma(C,H) + beta(C,H)*[EH(H) - SUM(CP, PQ(CP)*gamma(CP,H))] ;

 INVDEM(C)..     QINV(C) =E= IADJ*qinvbar(C) ;
 
 GOVDEM(C)..     QG(C)   =E= GADJ*qgbar(C) ;
 
 EGDEF..         EG =E= SUM(C, PQ(C)*QG(C)) ;

 YGDEF..         YG =E= ETAX + FTAX + HTAX + MTAX + PTAX + STAX ;

* Taxes
 ETAXEQ..        ETAX =E= SUM(C$CE(C), TE(C)*EXR*PWE(C)*QE(C)) ;
 
 FTAXEQ..        FTAX =E= SUM((f,a), TFA(F,A)*WF(F)*WFDIST(F,A)*QF(F,A)) ;
 
 HTAXEQ..        HTAX =E= SUM(H, TY(H)*YH(H)) ;
 
 MTAXEQ..        MTAX =E= SUM(C$CM(C), TM(C)*EXR*pwm(C)*QM(C)) ;
 
 PTAXEQ..        PTAX =E= SUM(A, TA(A)*PA(A)*QA(A)) ;
 
 STAXEQ..        STAX =E= SUM(C, TQ(C)*PQ(C)*QQ(C)) ;                             
                  
 GOVBAL..        YG =E= EG + GSAV;

*SYSTEM CONSTRAINT BLOCK

 COMEQUIL(C)$(CD(C) OR CM(C))..
                   QQ(C) =E= SUM(A, QINTCA(C,A)) + SUM(H, QH(C,H))
                           + QG(C) + QINV(C) ;
 
 FACEQUIL(F)..   SUM(A, QF(F,A)) =E= QFS(F);

 CURACCBAL..     SUM(C$CM(C),  pwm(C)*QM(C)) =E=
                             + SUM(C$CE(C), PWE(C)*QE(C)) 
                             + FSAV ;

 SAVINVBAL..     SUM(H, MPS(H)*(1 - TY(H))*YH(H)) + GSAV
                  + EXR*FSAV      =E=
                   SUM(C, PQ(C)*QINV(C)) + WALRAS ;
                   
*For balanced macro closure

 TABSEQ..        TABS =E= SUM((C,H), PQ(C)*QH(C,H)) +
                          SUM(C, PQ(C)*(QG(C) + QINV(C))) ;

 INVABEQ..       INVSHR*TABS =E= SUM(C, PQ(C)*QINV(C)) ;

 GDABEQ..        GOVSHR*TABS =E= SUM(C, PQ(C)*QG(C)) ;
          
 MPSDEF(H)..     MPS(H) =E= MPSBAR(H)*(1 + MPSADJ*mps01(H)) ;            

 CPIDEF..        CPI    =E= SUM(C, cwts(C)*PQ(C)) ;

 DPIDEF..        DPI    =E= SUM(C, dwts(C)*PD(C)) ;
$OffFold
     
* Include equations in the Model

$OnFold
MODEL

 UNI   MCP version with variable-equation association
/ 
 PMDEF  
 PEDEF 
 PQDEF 
 PXDEF  
 PADEF  
 PVADEF
 PRODFN
 FACDEM
 INTDEM
 OUTPUTFN
 CET
 CET2A
 CET2B
 ESUPPLY
 EDEMAND
 COSTMIN
 ARMINGTON
 ARMINGTON2A
 ARMINGTON2B
 YFDEF
 HHDINC
 EHDEF
 HHDEM
 INVDEM
 GOVDEM
 EGDEF
 YGDEF     
 GOVBAL  
 COMEQUIL
 FACEQUIL
 CURACCBAL
 SAVINVBAL
 TABSEQ 
 INVABEQ
 GDABEQ 
 MPSDEF    
 CPIDEF
 DPIDEF 
* Taxes
 ETAXEQ         
 FTAXEQ         
 HTAXEQ         
 MTAXEQ         
 PTAXEQ         
 STAXEQ         
/
 ;
 
$OffFold

*5. PARAMETER ESTIMATION AND CALIBRATION
$onText

 Given the transactions data in the Social Accounting Matrix (SAM), the selected functions used
 to represent behavior, and the elasticities of substitution or transformation, one needs
 to calibrate the parameters in the functions.
 
$offText

$INCLUDE 1Calibration-UNI.inc

*6. VARIABLE INITIALIZATION 
$OnFold

*VARIABLE NITIALIZATION===================
 
 CPI.L         = CPI0;
 DPI.L         = DPI0;
 EG.L          = EG0;
 EH.L(H)       = EH0(H);
 EXR.L         = EXR0;       
 FSAV.L        = FSAV0;
 GADJ.L        = GADJ0;
 GOVSHR.L      = GOVSHR0;
 GSAV.L        = GSAV0;
 IADJ.L        = IADJ0;
 INVSHR.L      = INVSHR0;
 MPS.L(H)      = MPS0(H);
 MPSADJ.L      = MPSADJ0;
 PA.L(A)       = PA0(A);
 PD.L(C)       = PD0(C);
 PE.L(C)       = PE0(C);
 PM.L(C)       = PM0(C);
 PQ.L(C)       = PQ0(C);
 PVA.L(A)      = PVA0(A);
 PWE.L(C)      = PWE0(C) ;
 PX.L(C)       = PX0(C);
 QA.L(A)       = QA0(A);
 QD.L(C)       = QD0(C);
 QE.L(C)       = QE0(C);
 QF.L(F,A)     = QF0(F,A);
 QFS.L(F)      = QFS0(F);
 QG.L(C)       = QG0(C);
 QH.L(C,H)     = QH0(C,H);
 QINTCA.L(C,A) = QINTCA0(C,A);
 QINV.L(C)     = QINV0(C);
 QM.L(C)       = QM0(C);
 QQ.L(C)       = QQ0(C);
 QX.L(C)       = QX0(C);
 TABS.L        = TABS0;
 TA.L(A)       = TA0(A) ;
 TE.L(C)       = TE0(C) ;
 TFA.L(F,A)    = TFA0(F,A) ;
 TM.L(C)       = TM0(C) ;
 TQ.L(C)       = TQ0(C) ;
 TY.L(H)       = TY0(H) ;
 WALRAS.L      = WALRAS0 ;
 WF.L(F)       = WF0(F);
 WFDIST.L(F,A) = WFDIST0(F,A);
 YF.L(F)       = YF0(F);
 YG.L          = YG0;
 YH.L(H)       = YH0(H);
 
* Taxes
 ETAX.L        = ETAX0 ;
 FTAX.L        = FTAX0 ;
 HTAX.L        = HTAX0 ;
 MTAX.L        = MTAX0 ;
 PTAX.L        = PTAX0 ;
 STAX.L        = STAX0 ;
 
$OffFold 
 
*7. MODEL CLOSURES
$OnFold
*SELECT CLOSURES===================================================

$ontext
select the closure settings by assigning a number to the parameter for each closure type 

Savings-Investment Closure options
  if SICLOS  = 1, savings rates fixed, investment adjusts (Default)
  if SICLOS  = 2, investment fixed, savings rates adjust
  
Government Closure options
  if GOVCLOS = 1, government demand for commodities is fixed (default)
  if GOVCLOS = 2, government demand for commodities adjusts
  if GOVCLOS = 3, government demand for commodites and government savings are fixed;
  direct inocme taxes on the single household adjusts
  
Foreign Exchange Market Closure options
  if ROWCLOS = 1, FSAV fixed, exchange rate is flexible
  if ROWCLOS = 2, Exchange rate is fixed, foreign savings (FSAV) is flexible
$offtext

Parameter
 SICLOS   savings-investment closure       /1/
 GOVCLOS  government expenditure closure   /2/
 ROWCLOS  foreign exchange market closure  /1/
 ;
 
* SAVINGS-INVESTMENT CLOSURE
IF(SICLOS EQ 1, 
*Savings-driven investment (neoclassical) -- the scalar on investment, IADJ, is flexible, permitting
*investment quantities and the investment value to adjust to the savings available

 MPSADJ.FX   = MPSADJ0;
 
 IADJ.LO     = -INF;
 IADJ.UP     = +INF;
 IADJ.L      = IADJ0;

 );

IF(SICLOS EQ 2,
*Investment-driven savings (Johansen) -- the marginal propensity to save adjusts for all households
*permitting the savings value to adjust to the investment expenditure.

 IADJ.FX     = IADJ0;
 
 MPSADJ.LO   = -INF;
 MPSADJ.UP   = +INF;
  
 );
  
* GOVERNNMENT CLOSURE
* Fix all tax variables to the base levels
 TA.FX(A)    = TA0(A) ;
 TE.FX(C)    = TE0(C) ;
 TFA.FX(F,A) = TFA0(F,A) ;
 TM.FX(C)    = TM0(C) ;
 TQ.FX(C)    = TQ0(C) ;
 TY.FX(H)    = TY0(H) ;

IF(GOVCLOS EQ 1,
*government consumption quantities are fixed, the government savings (i.e. the fiscal balance), GSAV, adjusts
 GADJ.FX   = GADJ0;
 
 GSAV.LO   = -inf ;
 GSAV.UP   = +inf ;
 );

IF(GOVCLOS EQ 2,
* government savings (i.e. the internaal balance), GSAV is fixed, government spending adjusts
* there is a scalar, GADJ that is applied to all government spending by commodity
 GADJ.LO   = -inf ;
 GADJ.UP   = +inf ;
 
 GSAV.FX   = GSAV0;

 );
 
IF(GOVCLOS EQ 3,
* tax replacement - direct income tax rate on the single household adjusts; policy
* changes that generate more government tax revenue are returned to the houseold in a
* "lump sum" payment

* government savings, (i.e. the internaal balance), GSAV is fixed
* governmet expenditure is fixed,
* government tax revenue adjusts through changes in the household income tax

 GADJ.FX   = GADJ0;
 
 GSAV.FX   = GSAV0;

 TY.LO("PRIV") = -inf ;
 TY.UP("PRIV") = +inf ;
 );
 
* FOREIGN EXCHANGE MARKET CLOSURE

IF(ROWCLOS EQ 1, 
*Foreign savings is fixed. A flexible exchange rate clears
*the current account of the balance of payments.
 FSAV.FX = FSAV0;
 EXR.LO  = -INF;
 EXR.UP  = +INF;
 EXR.L   = EXR0;
 );

IF(ROWCLOS EQ 2, 
*The exchange rate is fixed. Flexible foreign savings clears
*the current account of the balance of payments.
 EXR.FX   = EXR0;
 FSAV.LO  = -INF;
 FSAV.UP  = +INF;
 FSAV.L   = FSAV0;
 );
 
* FACTOR MARKETS
$ontext
 For each factor, there can be:
     Full Employment - the total factor supply is fixed and the factor payment (the wage) adjusts
     OR
     Unemployment - the total factor supply is flexible and the factor payment (the wage) is fixed.
 
 Full Employment is the default specification, to have unemployment, add a factor to the set FUE(f)
 in the excel workbook, on the sets tab

 When there is full employment of a factor, that factor can be fully mobile (long run)
 or activity-specific (short run)
 
 The default specification is that all factors are fully mobile
 
 To make a factor activity-specific, add that factor to the set FSF(f) in the excel workbook on the sets tab
 
 NOTE - a factor cannot be BOTH unemployed and activity-specific, if you assign the same factor to the set
 FUE(f) and FSF(f), the model treats the factor as unemployed

$offtext

Parameter
 FCLOS(F)  Factor market closure assignment 
 FSR(F)    Short run with activity specific factor

 ;
 
* If FCLOS(F) = 1, Full employment
* if FCLOS(F) = 2, Unemployment 
 FCLOS(F)         = 1 ;
 FCLOS(f)$FUE(f)  = 2 ;
  
 
* DO NOT MAKE ALL FACTORS Activity-Specific or the model will not
* solve for a simulation (no output can change because no factor can change production
* activity!)

 FSR(f) = 0 ;
 
 LOOP[F,
  WFDIST.FX(F,A) = WFDIST0(F,A) ;
  
  IF(FCLOS(F) = 1,
   QFS.FX(F)  = QFS0(F);
   QFS.L(F)   = QFS0(F);
   WF.LO(F)   = -INF;
   WF.UP(F)   = +INF;
   WF.L(F)    = WF0(F);
   );
   
  
* Short Run: Activity-Specific Factor
* since the factor cannot change activities, the factor market clearing condition for
* the activity-specific factor is redundant, therefore, fix the factor payment
* this option must be run in the full employment version of the model
 
* IF((FCLOS(F) = 1 AND FSR(F)= 1),
 IF((FCLOS(F) = 1 AND FSF(F)),
  QF.FX(F,A)      = QF0(F,A) ;
  WFDIST.LO(F,A)  = -INF ;
  WFDIST.UP(F,A)  = +INF ;
  WF.FX(F)        = WF0(F) ;
  );
  
 
$Ontext
  QF.FX('F-CAPITAL',A)      = QF0('F-CAPITAL',A) ;
  WFDIST.LO('F-CAPITAL',A)  = -INF ;
  WFDIST.UP('F-CAPITAL',A)  = +INF ;
  WF.FX('F-CAPITAL')        = WF0('F-CAPITAL') ;
  );
  
$offtext
  
  IF(FCLOS(F) = 2,
   QFS.LO(F)  = -INF;
   QFS.UP(F)  = +INF;
   QFS.L(F)   = QFS0(F);
   WF.FX(F)   = WF0(F);
   WF.L(F)    = WF0(F);
   );
 
*end F loop
 ] ;

*Fix all variables with no data and hence should be dropped from the model
 PM.FX(C)$CMN(C)              = 0 ;
 PE.FX(C)$CEN(C)              = 0 ;
 PQ.FX(C)$(CDN(C) AND CMN(C)) = 0 ;

 QM.FX(C)$CMN(C)              = 0 ;
 QE.FX(C)$CEN(C)              = 0 ;

 QF.FX(F,A)$(NOT AFA(F,A))    = 0 ;
 
 QQ.FX(C)$(CDN(C) AND CMN(C)) = 0 ;
 QD.FX(CDN)                   = 0 ;
 PD.FX(CDN)                   = 0 ;
 
 QH.FX(C,H)$(SAM(C,H) EQ 0)   = 0 ;
 
*Numeraire fixed (either CPI or DPI)
 CPI.FX                       = CPI0 ;
*DPI.FX                       = DPI0 ;

* Fix PWE when there is no export demand equation

 PWE.FX(C)$CEDN(C)            = PWE0(C);

$OffFold

*8. MODEL SOLUTION
$OnFold

*SOLVE STATEMENT FOR BASE========================================

options iterlim=1000,limrow=0,limcol=0,solprint=off;

*Choice of solvers
 Option MCP=PATH ;
 
 UNI.HOLDFIXED   = 1;
 UNI.TOLINFREP   = .00001 ;
 
 SOLVE UNI USING MCP ;
$OffFold


* 9. REPORT BASE MODEL
$OnFOld
*Calculate summary of base solution data
$include 1repbase-uni.inc
$OffFold
  
*****************************************************************************************
*                       EXPERIMENTS     EXPERIMENTS    EXPERIMENTS                      *
*****************************************************************************************

* IMPORTANT NOTE!!!
* When using the experiments below, you will need to the change set elements
* defined in quotes, to sector and factor names used in your SAM


*EXPERIMENT - Change a Sales Tax
$OnFold
display TQ0;
* TQ.FX(C) = TQ0(C) + 0.2 ;
* TQ.FX("C-SER") = TQ0("C-SER") * 1.2 ;
* TQ.FX("C-MFG") = TQ0("C-MFG") * 3 ;
display TQ.L;
$offFold

*EXPERIMENT - Change an Import Tariff
$OnFold
display TM0;
* TM.FX(C) = TM0(C) + 0.2 ;
* TM.FX("C-SER") = TM0("C-SER") * 1.5 ;
* TM.FX("C-MFG") = TM0("C-MFG") *1.05 ;
display TM.l;
$offFold

*EXPERIMENT - Change Total Factor Productivity
$onFold
Display alphaa;
*alphaa(A) =alphaa(A) * 1.05;
*alphaa("A-SER") = alphaa("A-SER") * 1.05
*alphaa("A-MFG") = alphaa("A-MFG") * 1.02;
Display alphaa; 
$OffFold

*EXPERIMENT - Change A World Import Price
$onFold

* define initial pwm value here and then use pwm0 to generate percent change in the results code
Parameter PWM0(c) Initial pwm value;
pwm0(c) = pwm(c) ;

Display pwm;
*pwm(c)= pwm0(c) * 1.05;
*pwm("c-mfg")= pwm("c-mfg") * 1.05;
Display pwm; 
$OffFold

*EXPERIMENT - Change a Factor Supply
$OnFold
Display QFS.l,ESUBVA;
* QFS.FX(F) = QFS0(F) * 1.02 ;
* QFS.FX("F-Capital") = QFS0("F-Capital") * 1.02 ;
* QFS.FX("F-Labor") = QFS0("F-Labor") * 1.02 ;
Display QFS.l;
$offFold

*EXPERIMENT - Consumer Taste Shift 
$OnFold
* This experiment changes beta, the marginal budget share
* There are 2 steps
*(1) Go to "Define NewCom" to define the commodities whose budget share changes
*(2) Go to "Define Experiment" to redefine a marginal budget share


* DEFINE NEWCOM(c) -- the commodity whose budget share changes
$ontext
Sets
  newcom(c)  commodity with change in consumer expenditure share
       /c-SER/
        
  newcomn(c)  commodities with no change in consumer expenditure share;
  newcomn(c) = not newcom(c)
  ;
  ALIAS(newcomn, ncomn) ;

Parameter  betab(c,h)          base beta marginal budget shares
           betat(h)            beta share sum check
           pc_beta(c,h)        percent change in beta share
           netbeta(c,h)        budget shares total net of newcom
           ckbeta(h)           sum of weighted budget shares
            ;
  betab(c,h)      = beta(c,h);
  
Display beta,betab;

* DEFINE EXPERIMENT
* Remove star from next line of code. Specify percent change in budget share of newcom commodity
* beta(newcom,h)  = .90 * betab(newcom,h);

* Rebalance remaining budget shares
  netbeta(newcomn,h) = betab(newcomn,h)/ sum(ncomn,betab(ncomn,h)) ;
  ckbeta(h) = sum(newcomn, netbeta(newcomn,h));
  netbeta(newcomn,h) = netbeta(newcomn,h)/ckbeta(h);
  beta(newcomn,h) = betab(newcomn,h)
              +  sum(newcom, betab(newcom,h) - beta(newcom,h))
              *  netbeta(newcomn,h)
       ;

* Check that rebalanced budget shares sum to one
   betat(h) = sum(c, beta(c,h));
   pc_beta(c,h) = (beta(c,h)- betab(c,h))/betab(c,h) * 100;
   
*recalculate budget shares in household spending
 BUDSHR(C,H)    = (PQ.l(c) * QH.l(C,H))/ SUM(CP, PQ.l(C) *QH.l(CP,H));

   display betab,netbeta,ckbeta,betat,pc_beta,newcom;
   display beta,frisch,gamma;
   
$offtext
$OffFold
  
 SOLVE UNI USING MCP;

$include UNI-Restab6.gms

























































































































































































































































































































































































































































